/*
 * MapCollection.h
 *
 * Created 8/19/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/19/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef MAP_COLLECTION_H
 #define MAP_COLLECTION_H
 
 template <typename K, typename V, typename COMP = std::less<K>> class MapCollection;
 
 #include <map>
 
 /**
  * Class specification for MapCollection, a generic collection class implementation using a map
  */
 template <typename K, typename V, typename COMP = std::less<K>>
 class MapCollection : private std::map<K, V, COMP>
 {
 // Public Encapsulated Classes
 public:
    class iterator;
 
 // Data Members
 protected:
    typedef std::map<K, V, COMP> GenericMap;
    
 // Local Functions
 public:
    MapCollection();
    MapCollection( const MapCollection<K, V, COMP>& mc );
    virtual ~MapCollection();
    inline MapCollection<K, V, COMP>& operator=( const MapCollection<K, V, COMP>& mc );
    virtual inline typename MapCollection<K, V, COMP>::iterator begin();
    virtual inline typename MapCollection<K, V, COMP>::iterator end();
    virtual inline void add( const K& key, const V& value );
    virtual inline void clear();
    virtual inline typename MapCollection<K, V, COMP>::iterator find( const K& key );
    inline V get( const K& key );
    virtual inline void remove( const K& key );
 
 // Private Functions
 private:
    
 // Friend Functions
 public:
    
 // Public Encapsulated Classes
 public:
    /**
     *  Class specification for MapCollection<K, V, COMP>::iterator
     */
    virtual class iterator
    {
    friend class MapCollection<K,V,COMP>; // for invoking private functions
    
    // Data members
    private:
        typename GenericMap::iterator _pair_Itr;
    
    // Local Functions
    public:
        iterator() { }
        iterator( const iterator& itr ) : _pair_Itr( itr._pair_Itr ) { }
        ~iterator() { }
        inline iterator& operator=( const iterator& itr ) { _pair_Itr = itr._pair_Itr; return *this; }
        inline bool operator==( const iterator& itr ) { return _pair_Itr == itr._pair_Itr; }
        inline bool operator!=( const iterator& itr ) { return _pair_Itr != itr._pair_Itr; }
        inline V& operator*() const { return (*this->_pair_Itr).second; } // dereference operator
        inline V& operator->() const { return (*this->_pair_Itr).second; } // member access operator
        inline iterator& operator++() { ++_pair_Itr; return *this; } // post-increment
        inline iterator& operator++(int) { ++_pair_Itr; return *this; } // pre-increment
        inline iterator& operator--() { --_pair_Itr; return *this; } // post-decrement
        inline iterator& operator--(int) { --_pair_Itr; return *this; } // pre-decrement
        
    // Private Functions
    private:
        iterator( const typename GenericMap::iterator& pair_Itr ) : _pair_Itr( pair_Itr ) { }
    };
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename K, typename V, typename COMP>
 MapCollection<K, V, COMP>::MapCollection()
                           : GenericMap()
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename K, typename V, typename COMP>
 MapCollection<K, V, COMP>::MapCollection( const MapCollection<K, V, COMP>& mc )
                           : GenericMap( mc )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename K, typename V, typename COMP>
 MapCollection<K, V, COMP>::~MapCollection()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified MapCollection to this MapCollection.
  *
  * @param (const MapCollection<K, V, COMP>&) mc
  * @return MapCollection<K, V, COMP>&
  */
 template <typename K, typename V, typename COMP>
 inline MapCollection<K, V, COMP>& MapCollection<K, V, COMP>::operator=( const MapCollection<K, V, COMP>& mc )
 {
    GenericMap::operator=( mc );
    
    return *this;
 }
 
 /**
  * begin() returns the beginning iterator of this collection.
  *
  * @return MapCollection<K, V, COMP>::iterator
  */
 template <typename K, typename V, typename COMP>
 inline typename MapCollection<K, V, COMP>::iterator MapCollection<K, V, COMP>::begin()
 {
    return GenericMap::begin();//MapCollection<K, V, COMP>::iterator( GenericMap::begin() );
 }
 
 /**
  * end() returns the ending iterator of this collection.
  *
  * @return MapCollection<K, V, COMP>::iterator
  */
 template <typename K, typename V, typename COMP>
 inline typename MapCollection<K, V, COMP>::iterator MapCollection<K, V, COMP>::end()
 {
    return GenericMap::end();//MapCollection<K, V, COMP>::iterator( GenericMap::end() );
 }
 
 /**
  * add() adds the specified value to this MapCollection, using the specified key.
  *
  * @param (const K&) key
  * @param (const V&) value
  */
 template <typename K, typename V, typename COMP>
 inline void MapCollection<K, V, COMP>::add( const K& key, const V& value )
 {
     GenericMap::insert( GenericMap::value_type( key, value ) );
 }
 
 /**
  * clear() removes all entries from this MapCollection, such that
  * this MapCollection will be empty.
  */
 template <typename K, typename V, typename COMP>
 inline void MapCollection<K, V, COMP>::clear()
 {
    GenericMap::clear();
 }
 
 /**
  * find() returns the iterator pointing to the value having the matching
  * specified key in this collection; if the value matching the specified
  * key is not found, an iterator equivalent to MapCollection<K, V, COMP>::end()
  * is returned.
  *
  * @return MapCollection<K, V, COMP>::iterator
  */
 template <typename K, typename V, typename COMP>
 inline typename MapCollection<K, V, COMP>::iterator MapCollection<K, V, COMP>::find( const K& key )
 {
    return GenericMap::find( key );//MapCollection<K, V, COMP>::iterator( GenericMap::find( key ) );
 }
 
 /**
  * get() returns a pointer to the Object that has the specified key.
  *
  * @param (K&) key
  * @return V
  */
 template <typename K, typename V, typename COMP>
 inline V MapCollection<K, V, COMP>::get( const K& key )
 {
    GenericMap::iterator pair_Itr( GenericMap::find( key ) );
    
    // if the key does not match any value in _values
    //if ( pair_Itr == GenericMap::end() )
        //return NULL;
    //else // the key matches a pair in this GenericMap
        return pair_Itr->second;
 }
 
 /**
  * remove() removes the value that has the specified key from this MapCollection.
  *
  * @param (const K&) key
  */
 template <typename K, typename V, typename COMP>
 inline void MapCollection<K, V, COMP>::remove( const K& key )
 {
    GenericMap::const_iterator pair_Itr( GenericMap::find( key ) );
    
    //if ( pair_Itr == GenericMap::end() )
        //return NULL;
    //else // the key matches a pair in this GenericMap
        GenericMap::erase( pair_Itr );
 }
 
 /** FRIEND FUNCTIONS **/
 
 #endif // MAP_COLLECTION_H